<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ItemMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\Facades\DataTables;

class ItemMasterController extends Controller
{
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $query = ItemMaster::select([
                'id', 'ItemCode', 'Description', 'UOM',
                'Item_No', 'ConvFactor', 'Base_UOM',
                'BusinessUnitid', 'CompanyId', 'IsWeightedItem'
            ])->orderBy('id', 'desc');

            return DataTables::eloquent($query)->make(true);
        }
        return view('admin.items.index');
    }

    /** Replace SQLite file with current DB records */
    public function replaceItemMaster(Request $request)
    {
        $request->validate([
            'sqlite_file' => 'required|file',
        ]);

        if (!preg_match('/\.(sqlite|db|sqlite3)$/i', $request->file('sqlite_file')->getClientOriginalName())) {
            return back()->withErrors(['sqlite_file' => 'Please upload a valid SQLite database file.']);
        }

        $uploadedFile = $request->file('sqlite_file');
        $originalName = $uploadedFile->getClientOriginalName(); // Keep original name
        $uploadedPath = $uploadedFile->storeAs('tmp', 'uploaded.sqlite');
        $sqlitePath = storage_path('app/private/' . $uploadedPath);
        Log::info('Uploaded SQLite path: '.$sqlitePath);

        try {
            $pdo = new \PDO("sqlite:$sqlitePath");
            $pdo->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);

            $pdo->exec("
                CREATE TABLE IF NOT EXISTS ItemMaster (
                    ItemCode TEXT,
                    Description TEXT,
                    UOM TEXT,
                    Price TEXT,
                    Cost TEXT,
                    Inventory TEXT,
                    QtyInPO TEXT,
                    QtyInSO TEXT,
                    QtyInTO TEXT,
                    Item_No TEXT,
                    VariantCode TEXT,
                    QtyinTransferIn TEXT,
                    QtyinTransferOut TEXT,
                    ConvFactor TEXT,
                    Base_UOM TEXT,
                    BusinessUnitid TEXT,
                    CompanyId TEXT,
                    IsWeightedItem TEXT
                )
            ");

            $columns = $pdo->query("PRAGMA table_info(ItemMaster)")->fetchAll(\PDO::FETCH_COLUMN, 1);

            if (!in_array('IsWeightedItem', $columns)) {
                $pdo->exec("ALTER TABLE ItemMaster ADD COLUMN IsWeightedItem TEXT");
            }

            $pdo->beginTransaction();
            $pdo->exec('DELETE FROM ItemMaster');

            // 📊 Fetch Laravel data
            $items = ItemMaster::all();
            $stmt = $pdo->prepare("
                INSERT INTO ItemMaster (
                    ItemCode, Description, UOM, Price, Cost, Inventory, QtyInPO,
                    QtyInSO, QtyInTO, Item_No, VariantCode, QtyinTransferIn,
                    QtyinTransferOut, ConvFactor, Base_UOM, BusinessUnitid, CompanyId, IsWeightedItem
                )
                VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)
            ");

            foreach ($items as $item) {
                $stmt->execute([
                    $item->ItemCode,
                    $item->Description,
                    $item->UOM,
                    $item->Price,
                    $item->Cost,
                    $item->Inventory,
                    $item->QtyInPO,
                    $item->QtyInSO,
                    $item->QtyInTO,
                    $item->Item_No,
                    $item->VariantCode,
                    $item->QtyinTransferIn,
                    $item->QtyinTransferOut,
                    $item->ConvFactor,
                    $item->Base_UOM,
                    $item->BusinessUnitid,
                    $item->CompanyId,
                    $item->IsWeightedItem
                ]);
            }

            // 💾 Commit changes
            $pdo->commit();
            $pdo = null;

            //session()->flash('success', 'ItemMaster table successfully updated in the uploaded SQLite file!');

            // Return the download response
            return response()->download($sqlitePath, $originalName)->deleteFileAfterSend(true);

        } catch (\Exception $e) {
            if (isset($pdo)) {
                $pdo->rollBack();
            }
            Log::error('SQLite update failed: '.$e->getMessage());
            return back()->withErrors(['sqlite_file' => 'Failed to update SQLite: '.$e->getMessage()]);
        }
    }
}
